﻿/*
 * Copyright 2020 faddenSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
using System;
using System.Collections.ObjectModel;

using PluginCommon;

namespace Bomber {
    public class VisTutorial5 : MarshalByRefObject, IPlugin, IPlugin_Visualizer {
        // IPlugin
        public string Identifier {
            get { return "Bomber Visualizer"; }
        }
        private IApplication mAppRef;
        private byte[] mFileData;

        // Visualization identifiers; DO NOT change or projects that use them will break.
        private const string VIS_PIXEL_LIST = "bomber-pixel-list";

        private const string P_OFFSET = "offset";
        private const string P_COUNT = "count";

        // Visualization descriptors.
        private VisDescr[] mDescriptors = new VisDescr[] {
            new VisDescr(VIS_PIXEL_LIST, "Bomber Pixel List", VisDescr.VisType.Bitmap,
                new VisParamDescr[] {
                    new VisParamDescr("File offset (hex)",
                        P_OFFSET, typeof(int), 0, 0x00ffffff, VisParamDescr.SpecialMode.Offset, 0),
                    new VisParamDescr("Count",
                        P_COUNT, typeof(int), 1, 256, 0, 1),
                })
        };


        // IPlugin
        public void Prepare(IApplication appRef, byte[] fileData, AddressTranslate addrTrans) {
            mAppRef = appRef;
            mFileData = fileData;
        }

        // IPlugin
        public void Unprepare() {
            mAppRef = null;
            mFileData = null;
        }

        // IPlugin_Visualizer
        public VisDescr[] GetVisGenDescrs() {
            if (mFileData == null) {
                return null;
            }
            return mDescriptors;
        }

        // IPlugin_Visualizer
        public IVisualization2d Generate2d(VisDescr descr,
                ReadOnlyDictionary<string, object> parms) {
            switch (descr.Ident) {
                case VIS_PIXEL_LIST:
                    return GenerateBitmap(parms);
                default:
                    mAppRef.ReportError("Unknown ident " + descr.Ident);
                    return null;
            }
        }

        private IVisualization2d GenerateBitmap(ReadOnlyDictionary<string, object> parms) {
            int offset = Util.GetFromObjDict(parms, P_OFFSET, 0);
            int count = Util.GetFromObjDict(parms, P_COUNT, 1);

            if (offset < 0 || offset >= mFileData.Length ||
                    count <= 0 || count > 256) {
                mAppRef.ReportError("Invalid parameter");
                return null;
            }

            int lastOffset = offset + count * 2;
            if (lastOffset >= mFileData.Length) {
                mAppRef.ReportError("Data runs off end of file (last offset +" +
                    lastOffset.ToString("x6") + ")");
                return null;
            }

            // Compute bounds.
            int top, bottom, left, right;
            top = left = 257;
            bottom = right = -1;
            for (int i = 0; i < count; i++) {
                int col = mFileData[offset + i * 2];
                int row = mFileData[offset + i * 2 + 1];

                if (left > col ) {
                    left = col;
                }
                if (right < col) {
                    right = col;
                }
                if (top > row) {
                    top = row;
                }
                if (bottom < row) {
                    bottom = row;
                }
            }
            mAppRef.DebugLog("Pixel bounds: left=" + left + " top=" + top +
                " right=" + right + " bottom=" + bottom);

            VisBitmap8 vb = new VisBitmap8(right - left + 1, bottom - top + 1);
            SetPalette(vb);

            // Render.
            for (int i = 0; i < count; i++) {
                int col = mFileData[offset + i * 2];
                int row = mFileData[offset + i * 2 + 1];

                vb.SetPixelIndex(col - left, row - top, (byte)Color.Solid);
            }

            return vb;
        }

        private enum Color : byte {
            Transparent = 0,
            Solid = 1,
        }

        private void SetPalette(VisBitmap8 vb) {
            vb.AddColor(0, 0, 0, 0);                // 0=transparent
            vb.AddColor(0xff, 0x20, 0x20, 0xff);    // 1=solid
        }
    }
}
