﻿/*
 * Copyright 2021 faddenSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;

using PluginCommon;

namespace Deathmaze5000 {
    /// <summary>
    /// Deathmaze 5000 floor map rendering.
    /// </summary>
    public class VisDM5K : MarshalByRefObject, IPlugin, IPlugin_Visualizer {
        // IPlugin
        public string Identifier {
            get { return "Deathmaze 5000 visualizer"; }
        }
        private IApplication mAppRef;
        private byte[] mFileData;
        private AddressTranslate mAddrTrans;

        // Visualization identifiers; DO NOT change or projects that use them will break.
        private const string VIS_GEN_MAZE = "dm5k-maze";

        private const string P_FLOOR = "floor";

        private const int NUM_FLOORS = 5;
        private const int BYTES_PER_FLOOR = 33;
        private const int FLOOR_DATA_OFFSET = 0x37fb;     // addr $6000
        private const int OBJ_DATA_OFFSET = 0x395e;
        private const int NUM_OBJS = 24;

        private const int CELL_SIZE = 24;

        // Visualization descriptors.
        private VisDescr[] mDescriptors = new VisDescr[] {
            new VisDescr(VIS_GEN_MAZE, "Deathmaze 5000 floor map", VisDescr.VisType.Bitmap,
                new VisParamDescr[] {
                    new VisParamDescr("Floor",
                        P_FLOOR, typeof(int), 1, 5, 0, 1),
                }),
        };


        // IPlugin
        public void Prepare(IApplication appRef, byte[] fileData, AddressTranslate addrTrans) {
            mAppRef = appRef;
            mFileData = fileData;
            mAddrTrans = addrTrans;
        }

        // IPlugin
        public void Unprepare() {
            mAppRef = null;
            mFileData = null;
            mAddrTrans = null;
        }

        // IPlugin_Visualizer
        public VisDescr[] GetVisGenDescrs() {
            // We're using a static set, but it could be generated based on file contents.
            // Confirm that we're prepared.
            if (mFileData == null) {
                return null;
            }
            return mDescriptors;
        }

        // IPlugin_Visualizer
        public IVisualization2d Generate2d(VisDescr descr,
                ReadOnlyDictionary<string, object> parms) {
            switch (descr.Ident) {
                case VIS_GEN_MAZE:
                    return GenerateMazeFloor(parms);
                default:
                    mAppRef.ReportError("Unknown ident " + descr.Ident);
                    return null;
            }
        }

        private IVisualization2d GenerateMazeFloor(ReadOnlyDictionary<string, object> parms) {
            int floor = Util.GetFromObjDict(parms, P_FLOOR, 0);

            // Do a couple of basic checks.
            if (floor < 1 || floor > NUM_FLOORS) {
                // should be caught by editor
                mAppRef.ReportError("Invalid parameter");
                return null;
            }
            if (FLOOR_DATA_OFFSET + NUM_FLOORS * BYTES_PER_FLOOR >= mFileData.Length) {
                mAppRef.ReportError("Invalid file");
                return null;
            }

            const int spacing = 1;
            const int glyphSize = 8;
            const int cWidth = CELL_SIZE + spacing;
            const int cHeight = CELL_SIZE + spacing;
            const int labelStripWidth = glyphSize * 2 + 1;
            const int labelStripHeight = glyphSize;
            const int cnSn = (CELL_SIZE - glyphSize) / 2;            // 8
            const int cnDb = (CELL_SIZE - glyphSize * 2) / 2;        // 4

            const int numCols = 11;
            const int numRows = 12;

            const int bitmapWidth = labelStripWidth + cWidth * numCols;
            const int bitmapHeight = labelStripHeight + cHeight * numRows;
            VisBitmap8 vb = new VisBitmap8(bitmapWidth, bitmapHeight);
            SetPalette(vb);

            // Draw labels 1-9.
            for (int i = 1; i < 10; i++) {
                char ch = (char)('0' + i);

                // left edge
                VisBitmap8.DrawChar(vb, ch,
                    1 + glyphSize / 2,
                    cnSn + (numRows - i) * cHeight,
                    (byte)Color.NumText, (byte)Color.NumBkgnd);
                // bottom edge
                VisBitmap8.DrawChar(vb, ch,
                    labelStripWidth + (i - 1) * cWidth + cnSn,
                    numRows * cHeight,
                    (byte)Color.NumText, (byte)Color.NumBkgnd);
            }
            // Draw labels 10-11 (just 10 across the bottom).
            for (int i = 10; i < 12; i++) {
                char ch0 = (char)('0' + i / 10);
                char ch1 = (char)('0' + i % 10);

                // left edge
                VisBitmap8.DrawChar(vb, ch0, 1,             cnSn + (numRows - i) * cHeight,
                    (byte)Color.NumText, (byte)Color.NumBkgnd);
                VisBitmap8.DrawChar(vb, ch1, 1 + glyphSize, cnSn + (numRows - i) * cHeight,
                    (byte)Color.NumText, (byte)Color.NumBkgnd);

                if (i >= numCols) {
                    continue;
                }

                // bottom edge
                VisBitmap8.DrawChar(vb, ch0,
                    labelStripWidth + (i - 1) * cWidth + cnDb,
                    numRows * cHeight,
                    (byte)Color.NumText, (byte)Color.NumBkgnd);
                VisBitmap8.DrawChar(vb, ch1,
                    labelStripWidth + (i - 1) * cWidth + cnDb + glyphSize,
                    numRows * cHeight,
                    (byte)Color.NumText, (byte)Color.NumBkgnd);
            }

            // Draw cells.  We draw an 11x11 grid, because the 11th row and column
            // form the top and right edge of the maze.
            for (int row = 1; row <= numRows; row++) {
                for (int col = 1; col <= numCols; col++) {
                    int xc = labelStripWidth + (col - 1) * cWidth;
                    int yc = (numRows - row) * cHeight;

                    bool hasSouth, hasWest;
                    GetWallData(floor, col, row, out hasSouth, out hasWest);

                    RenderCell(vb, xc, yc, hasSouth, hasWest);
                }
            }
            // Draw stuff inside cells.  Some of the stuff overlaps into adjacent cells,
            // possibly drawn on top of walls, which is why we have to draw the background
            // and walls for everything first.
            for (int row = 1; row <= numRows; row++) {
                for (int col = 1; col <= numCols; col++) {
                    int xc = labelStripWidth + (col - 1) * cWidth;
                    int yc = (numRows - row) * cHeight;

                    bool hasSouth, hasWest;
                    GetWallData(floor, col, row, out hasSouth, out hasWest);

                    int obj;
                    Facing facing;
                    Feature feature;
                    if ((facing = StartInCell(floor, col, row)) != Facing.Unknown) {
                        // Prioritize start 'S' over hole-in-roof 'H'.
                        VisBitmap8.DrawChar(vb, 'S',
                            xc + (hasWest ? 10 : 8), yc + 8,
                            (byte)Color.StartMarker, (byte)Color.CellBkgnd);
                    } else if (FeatureInCell(floor, col, row, out feature, out facing)) {
                        // Feature here, show a letter and optional facing indicator.
                        char featCh = '?';
                        bool drawFace = false;
                        Color color = Color.FeatMarker;
                        switch (feature) {
                            case Feature.Keyhole:
                                featCh = 'K'; drawFace = true; break;
                            case Feature.Elevator:
                                featCh = 'E'; drawFace = true; break;
                            case Feature.Pit:
                                featCh = 'P'; break;
                            case Feature.CeilingHole:
                                featCh = 'H'; break;
                            case Feature.PerfectSquare:
                                featCh = 'Q'; drawFace = true; break;
                            case Feature.Dog2:
                                featCh = 'D'; color = Color.HostileMarker; break;
                            case Feature.Bat:
                                featCh = 'V'; color = Color.HostileMarker; break;
                            case Feature.Guillotine:
                                featCh = 'G'; color = Color.HostileMarker; break;
                            case Feature.CalcPuzzle:
                                featCh = 'C'; color = Color.HostileMarker; break;
                            default:                    featCh = '?'; break;
                        }
                        VisBitmap8.DrawChar(vb, featCh,
                            xc + (hasWest ? 10 : 8), yc + 8,
                            (byte)color, (byte)Color.CellBkgnd);
                        if (drawFace) {
                            // Add a perpendicular line to show which way the feature faces,
                            // e.g. which side the elevator door is on.
                            int xStart, yStart, width, height;
                            switch (facing) {
                                case Facing.West:
                                    width = 8;
                                    height = 2;
                                    xStart = xc - width / 2 + 2;
                                    yStart = yc + CELL_SIZE / 2 - 1;
                                    break;
                                case Facing.North:
                                    width = 2;
                                    height = 8;
                                    xStart = xc + CELL_SIZE / 2;
                                    yStart = yc - height / 2;
                                    break;
                                case Facing.East:
                                    width = 8;
                                    height = 2;
                                    xStart = xc + CELL_SIZE - width / 2 + 3;
                                    yStart = yc + CELL_SIZE / 2 - 1;
                                    break;
                                case Facing.South:
                                    width = 2;
                                    height = 8;
                                    xStart = xc + CELL_SIZE / 2;
                                    yStart = yc + CELL_SIZE - height / 2 - 1;
                                    break;
                                default:
                                    width = 1;
                                    height = 1;
                                    xStart = xc;
                                    yStart = yc;
                                    break;
                            }
                            DrawRect(vb, xStart, yStart, width, height, color);
                        }
                    } else if ((obj = ObjectInCell(floor, col, row)) != 0) {
                        // Object here, show two-digit hex value.
                        DrawHexValue(vb,
                            xc + (hasWest ? 7 : 4), yc + (hasSouth ? 7 : 8),
                            obj, Color.ObjMarker, Color.CellBkgnd);
                    }
                }
            }

            return vb;
        }

        /// <summary>
        /// Renders one map cell.
        /// </summary>
        private void RenderCell(VisBitmap8 vb, int startX, int startY,
                bool hasSouth, bool hasWest) {
            const int wallWidth = 4;

            for (int yc = startY; yc < startY + CELL_SIZE; yc++) {
                for (int xc = startX; xc < startX + CELL_SIZE; xc++) {
                    byte color = (byte)Color.CellBkgnd;
                    if (hasSouth && yc > startY + CELL_SIZE - wallWidth) {
                        color = (byte)Color.Wall;
                    } else if (hasWest && xc < startX + wallWidth) {
                        color = (byte)Color.Wall;
                    }
                    vb.SetPixelIndex(xc, yc, color);
                }
            }
        }

        private void DrawRect(VisBitmap8 vb, int xc, int yc, int width, int height, Color color) {
            for (int row = yc; row < yc + height; row++) {
                for (int col = xc; col < xc + width; col++) {
                    vb.SetPixelIndex(col, row, (byte)color);
                }
            }
        }

        private readonly char[] HexValue = {
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
        };

        /// <summary>
        /// Draws a two-digit hex value.
        /// </summary>
        private void DrawHexValue(VisBitmap8 vb, int xc, int yc, int value,
                Color fgColor, Color bgColor) {
            if (value < 0 || value > 255) {
                mAppRef.ReportError("Internal error: bad hex value " + value);
                return;
            }
            VisBitmap8.DrawChar(vb, HexValue[value >> 4], xc, yc,
                (byte)fgColor, (byte)bgColor);
            VisBitmap8.DrawChar(vb, HexValue[value & 0x0f], xc + 8, yc,
                (byte)fgColor, (byte)bgColor);
        }

        private void GetWallData(int floor, int col, int row, out bool hasSouth, out bool hasWest) {
            const int bytesPerCol = 3;      // each column holds 12 rows, 2 bits per row

            int offset = FLOOR_DATA_OFFSET + (floor - 1) * BYTES_PER_FLOOR +
                (col - 1) * bytesPerCol + (row - 1) / 4;
            byte mask;
            switch ((row - 1) % 4) {
                case 0: mask = 0x80; break;
                case 1: mask = 0x20; break;
                case 2: mask = 0x08; break;
                case 3: mask = 0x02; break;
                default:
                    mAppRef.ReportError("Internal error: bad mask for row=" + row);
                    mask = 0;
                    break;
            }

            byte data = mFileData[offset];
            hasSouth = (data & mask) != 0;
            hasWest = (data & (mask >> 1)) != 0;
        }

        private int ObjectInCell(int floor, int col, int row) {
            int offset = OBJ_DATA_OFFSET;

            for (int i = 0; i < NUM_OBJS; i++) {
                if (floor == (int)mFileData[offset + i * 2] &&
                        col * 16 + row == (int)mFileData[offset + i * 2 + 1]) {
                    return i;
                }
            }
            return 0;
        }

        private enum Facing {
            Unknown = 0,
            West = 1,
            North = 2,
            East = 3,
            South = 4
        }

        private struct StartPosition {
            public int mFloor;
            public int mX;
            public int mY;
            public Facing mFacing;

            public StartPosition(int floor, int x, int y, Facing facing) {
                mFloor = floor;
                mX = x;
                mY = y;
                mFacing = facing;
            }
        }
        private StartPosition[] StartPositions = {
            new StartPosition(1, 10, 6, Facing.North),      // see $613d (initial values)
            new StartPosition(2, 3, 3, Facing.North),       // see $305c (charge)
            new StartPosition(3, 8, 5, Facing.East),        // see $0a91 (pit)
            new StartPosition(4, 1, 4, Facing.West),        // see $3a03 (elevator)
            new StartPosition(5, 3, 3, Facing.West),        // see $2e0a (calculator "two")
        };
        private Facing StartInCell(int floor, int col, int row) {
            foreach (StartPosition sp in StartPositions) {
                if (floor == sp.mFloor && col == sp.mX && row == sp.mY) {
                    return sp.mFacing;
                }
            }
            return Facing.Unknown;
        }

        // Visual features and special events.
        private enum Feature {
            Unknown = 0,
            Keyhole = 1,
            Elevator = 2,
            Pit = 4,
            CeilingHole = 5,
            PerfectSquare = 7,

            CalcPuzzle,
            Guillotine,
            Dog2,
            Bat,
        }
        private struct FeaturePosition {
            public int mFloor;
            public int mX;
            public int mY;
            public Facing mFacing;
            public Feature mFeature;

            public FeaturePosition(int floor, int x, int y, Facing facing, Feature feature) {
                mFloor = floor;
                mX = x;
                mY = y;
                mFacing = facing;
                mFeature = feature;
            }
        }
        private FeaturePosition[] FeaturePositions = {
            // Features, derived from the table at $60a5.
            new FeaturePosition(2, 3, 3, Facing.Unknown, Feature.CeilingHole),
            new FeaturePosition(2, 8, 5, Facing.Unknown, Feature.Pit),
            new FeaturePosition(2, 8, 7, Facing.West, Feature.Elevator),
            new FeaturePosition(3, 3, 4, Facing.East, Feature.Elevator),
            new FeaturePosition(3, 8, 5, Facing.Unknown, Feature.CeilingHole),
            new FeaturePosition(3, 7, 9, Facing.South, Feature.PerfectSquare),
            new FeaturePosition(4, 1, 5, Facing.South, Feature.Elevator),
            new FeaturePosition(4, 1, 10, Facing.Unknown, Feature.CeilingHole),
            new FeaturePosition(5, 3, 6, Facing.South, Feature.Elevator),
            new FeaturePosition(5, 4, 11, Facing.South, Feature.Keyhole),
            new FeaturePosition(5, 5, 11, Facing.South, Feature.Keyhole),
            new FeaturePosition(5, 6, 11, Facing.South, Feature.Keyhole),
            new FeaturePosition(5, 7, 11, Facing.South, Feature.Keyhole),
            new FeaturePosition(5, 8, 11, Facing.South, Feature.Keyhole),

            // Special events, from the code at $0a10.
            new FeaturePosition(1, 3, 3, Facing.Unknown, Feature.CalcPuzzle),
            new FeaturePosition(1, 6, 10, Facing.Unknown, Feature.Guillotine),
            new FeaturePosition(2, 5, 5, Facing.Unknown, Feature.Dog2),
            new FeaturePosition(5, 4, 4, Facing.Unknown, Feature.Bat),
        };
        private bool FeatureInCell(int floor, int col, int row, out Feature feature,
                out Facing facing) {
            foreach (FeaturePosition fp in FeaturePositions) {
                if (floor == fp.mFloor && col == fp.mX && row == fp.mY) {
                    feature = fp.mFeature;
                    facing = fp.mFacing;
                    return true;
                }
            }
            feature = Feature.Unknown;
            facing = Facing.Unknown;
            return false;
        }

        private enum Color : byte {
            Transparent = 0,            // gaps between cells
            NumText = 1,                // black for numbers around outside (foreground)
            NumBkgnd = 2,               // white for numbers around outside (background)
            CellBkgnd = 3,              // light grey for cell background
            Wall = 4,                   // near-black for walls
            ObjMarker = 5,              // green for objects
            FeatMarker = 6,             // blue for features (elevators, pits)
            HostileMarker = 7,          // red for hostiles (dog, bat, monster)
            StartMarker = 8,            // orange for start positions
        }

        private void SetPalette(VisBitmap8 vb) {
            vb.AddColor(0, 0, 0, 0);                // 0=transparent
            vb.AddColor(0xff, 0x00, 0x00, 0x00);    // 1=black
            vb.AddColor(0xff, 0xff, 0xff, 0xff);    // 2=white
            vb.AddColor(0xff, 0xf0, 0xf0, 0xf0);    // 3=light grey
            vb.AddColor(0xff, 0x10, 0x10, 0x10);    // 4=near black
            vb.AddColor(0xff, 0x00, 0x8f, 0x00);    // 5=dark green
            vb.AddColor(0xff, 0x00, 0x00, 0x8f);    // 6=dark blue
            vb.AddColor(0xff, 0x8f, 0x00, 0x00);    // 7=dark red
            vb.AddColor(0xff, 0xbf, 0x69, 0x00);    // 8=dark orange
        }
    }
}
