﻿/*
 * Copyright 2020 faddenSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Runtime.CompilerServices;

using PluginCommon;

namespace Epoch {
    /// <summary>
    /// Visualizer for shape data in Epoch for the Apple II.
    /// </summary>
    public class VisEpoch : MarshalByRefObject, IPlugin, IPlugin_Visualizer {
        private bool VERBOSE = false;

        // IPlugin
        public string Identifier {
            get { return "Epoch Shape Visualizer"; }
        }
        private IApplication mAppRef;
        private byte[] mFileData;

        // Visualization identifiers; DO NOT change or projects that use them will break.
        private const string VIS_EPOCH_SHAPE = "apple2-epoch-shape";

        private const string P_OFFSET = "offset";
        private const string P_OUTLINE_MODE = "outline-mode";
        private const string P_SCRAMBLE_COLORS = "scramble-colors";

        private const int BITMAP_DIM = 64;
        private static bool DOUBLE_WIDE = false;    // mildly broken, didn't really look good

        // Visualization descriptors.
        private VisDescr[] mDescriptors = new VisDescr[] {
            new VisDescr(VIS_EPOCH_SHAPE, "Epoch Shape", VisDescr.VisType.Bitmap,
                new VisParamDescr[] {
                    new VisParamDescr("File offset (hex)",
                        P_OFFSET, typeof(int), 0, 0x00ffffff, VisParamDescr.SpecialMode.Offset, 0),
                    new VisParamDescr("Draw outlines",
                        P_OUTLINE_MODE, typeof(bool), 0, 0, 0, false),
                    new VisParamDescr("Scramble colors",
                        P_SCRAMBLE_COLORS, typeof(bool), 0, 0, 0, false),
                })
        };


        // IPlugin
        public void Prepare(IApplication appRef, byte[] fileData, AddressTranslate addrTrans) {
            mAppRef = appRef;
            mFileData = fileData;
        }

        // IPlugin
        public void Unprepare() {
            mAppRef = null;
            mFileData = null;
        }

        // IPlugin_Visualizer
        public VisDescr[] GetVisGenDescrs() {
            if (mFileData == null) {
                return null;
            }
            return mDescriptors;
        }

        // IPlugin_Visualizer
        public IVisualization2d Generate2d(VisDescr descr,
                ReadOnlyDictionary<string, object> parms) {
            switch (descr.Ident) {
                case VIS_EPOCH_SHAPE:
                    return GenerateBitmap(parms);
                default:
                    mAppRef.ReportError("Unknown ident " + descr.Ident);
                    return null;
            }
        }

        enum ColorMode { Unknown = 0, Standard = 1, Dim = 2, Motley = 3 }

        private static HiResColors sMotleyColor = HiResColors.Black0;

        private abstract class Element {
            protected byte mColorA, mColorB;
            protected byte mColorChange;
            protected int mLeft, mTop, mRight, mBottom;
            protected IApplication mAppRef;

            public Element(IApplication appRef, byte colorA, byte colorB, byte colorChange,
                    short left, short top, short right, short bottom) {
                if (left > right) {
                    appRef.DebugLog("GLITCH: bad l/r order: l=" + left + " r=" + right);
                    right = left;
                }
                if (top > bottom) {
                    appRef.DebugLog("GLITCH: bad t/b order: t=" + top + " b=" + bottom);
                    bottom = top;
                }
                mAppRef = appRef;
                mColorA = colorA;
                mColorB = colorB;
                mColorChange = colorChange;
                // Epoch treats the screen like a square, so widths are effectively multiplied
                // by 280/192=1.4583.  (There's potentially some additional distortion because
                // the hi-res screen doesn't have square pixels, but I'm ignoring that.)
                const double kWidthAdj = 1.4583;
                mLeft = (int)(left * kWidthAdj);
                mTop = top;
                mRight = (int)(right * kWidthAdj);
                mBottom = bottom;
            }

            public void UpdateMinMax(ref int minX, ref int minY, ref int maxX, ref int maxY) {
                if (mLeft < minX) {
                    minX = mLeft;
                }
                if (mRight > maxX) {
                    maxX = mRight;
                }
                if (mTop < minY) {
                    minY = mTop;
                }
                if (mBottom > maxY) {
                    maxY = mBottom;
                }
            }

            protected int Scale(int val, double scale) {
                return (BITMAP_DIM / 2) + (int)Math.Round(val * scale);
            }

            public abstract void Draw(VisBitmap8 vb, ColorMode cmode, double scale,
                    bool outlineOnly);
        }

        private class EpochPoint : Element {
            public EpochPoint(IApplication appRef, byte colorA, byte colorB, byte colorChange,
                    short left, short top, short right, short bottom) :
                    base(appRef, colorA, colorB, colorChange, left, top, right, bottom) {
            }
            public override void Draw(VisBitmap8 vb, ColorMode cmode, double scale,
                    bool outlineOnly) {
                int left = Scale(mLeft, scale);
                int top = Scale(mTop, scale);
                int right = Scale(mRight, scale);
                int bottom = Scale(mBottom, scale);
                byte colorA = (byte)GetHiResColor(mColorA, cmode);

                if (left != right || top != bottom) {
                    mAppRef.DebugLog("GLITCH: type is point, shape is rect");
                    colorA = (byte)HiResColors.Whoops;
                }

                // ignore bottom/right
                vb.SetPixelIndex(left, top, colorA);
                if (DOUBLE_WIDE) {
                    vb.SetPixelIndex(left+1, top, colorA);
                    vb.SetPixelIndex(left, top+1, colorA);
                    vb.SetPixelIndex(left+1, top+1, colorA);
                }
            }
        }

        private class EpochLine : Element {
            public EpochLine(IApplication appRef, byte colorA, byte colorB, byte colorChange,
                    short left, short top, short right, short bottom) :
                    base(appRef, colorA, colorB, colorChange, left, top, right, bottom) {
            }
            public override void Draw(VisBitmap8 vb, ColorMode cmode, double scale,
                    bool outlineOnly) {
                int left = Scale(mLeft, scale);
                int top = Scale(mTop, scale);
                int right = Scale(mRight, scale);
                int bottom = Scale(mBottom, scale);
                byte colorA = (byte)GetHiResColor(mColorA, cmode);
                byte colorB = (byte)GetHiResColor(mColorB, cmode);

                if (!(left == right || top == bottom)) {
                    mAppRef.DebugLog("GLITCH: type is line, shape is rect");
                    colorA = (byte)HiResColors.Whoops;
                }

                if (left == right) {
                    // vertical line
                    int endCol = left;
                    if (DOUBLE_WIDE) {
                        endCol++;
                    }
                    for (int col = left; col <= endCol; col++) {
                        for (int row = top; row <= bottom; row++) {
                            byte color = colorA;
                            if (mColorChange != 0 && ((col & 0x01) ^ (row & 0x01)) != 0) {
                                color = colorB;
                            }
                            vb.SetPixelIndex(col, row, color);
                        }
                    }
                } else {
                    // horizontal line
                    int endRow = top;
                    if (DOUBLE_WIDE) {
                        endRow++;
                    }
                    for (int row = top; row <= endRow; row++) {
                        for (int col = left; col <= right; col++) {
                            byte color = colorA;
                            if (mColorChange != 0 && ((col & 0x01) ^ (row & 0x01)) != 0) {
                                color = colorB;
                            }
                            vb.SetPixelIndex(col, row, color);
                        }
                    }
                }
            }
        }

        private class EpochRect : Element {
            public EpochRect(IApplication appRef, byte colorA, byte colorB, byte colorChange,
                    short left, short top, short right, short bottom) :
                    base(appRef, colorA, colorB, colorChange, left, top, right, bottom) {
            }
            public override void Draw(VisBitmap8 vb, ColorMode cmode, double scale,
                    bool outlineOnly) {
                int left = Scale(mLeft, scale);
                int top = Scale(mTop, scale);
                int right = Scale(mRight, scale);
                int bottom = Scale(mBottom, scale);
                byte colorA = (byte)GetHiResColor(mColorA, cmode);
                byte colorB = (byte)GetHiResColor(mColorB, cmode);

                //mAppRef.DebugLog("Draw rect l=" + left + " t=" + top + " r=" + right +
                //    " b=" + bottom);

                if (top < 0 || left < 0) {
                    mAppRef.DebugLog("GLITCH: invalid top=" + top + " (mT=" + mTop +
                        ") left=" + left + " (mL=" + mLeft + ") scale=" + scale);
                }

                if (outlineOnly) {
                    for (int row = top; row <= bottom; row++) {
                        vb.SetPixelIndex(left, row, colorA);
                        vb.SetPixelIndex(right, row, colorA);
                    }
                    for (int col = left; col <= right; col++) {
                        vb.SetPixelIndex(col, top, colorA);
                        vb.SetPixelIndex(col, bottom, colorA);
                    }
                } else {
                    for (int row = top; row <= bottom; row++) {
                        for (int col = left; col <= right; col++) {
                            byte color = colorA;
                            if (mColorChange != 0 && ((col & 0x01) ^ (row & 0x01)) != 0) {
                                color = colorB;
                            }
                            vb.SetPixelIndex(col, row, color);
                        }
                    }
                }
            }
        }

        private IVisualization2d GenerateBitmap(ReadOnlyDictionary<string, object> parms) {
            const int HEADER_LEN = 24;
            const int ELEM_LEN = 16;
            const int MIN_LEN = HEADER_LEN + ELEM_LEN;

            int offset = Util.GetFromObjDict(parms, P_OFFSET, 0);
            bool outlineMode = Util.GetFromObjDict(parms, P_OUTLINE_MODE, false);
            bool scrambleColors = Util.GetFromObjDict(parms, P_SCRAMBLE_COLORS, false);

            if (offset < 0 || offset + MIN_LEN >= mFileData.Length) {
                mAppRef.ReportError("Invalid parameter");
                return null;
            }

            byte elemCount = mFileData[offset];
            if (offset + HEADER_LEN + elemCount * ELEM_LEN >= mFileData.Length) {
                mAppRef.ReportError("Shape runs off end of file");
                return null;
            }

            //
            // Start by gathering up all of the elements.
            //
            List<Element> elems = new List<Element>(elemCount);

            int elemOff = offset + HEADER_LEN;
            for (int i = 0; i < elemCount; i++) {
                byte kind = mFileData[elemOff + 0];
                byte colorA = mFileData[elemOff + 1];
                byte colorB = mFileData[elemOff + 2];
                byte colorFreq = mFileData[elemOff + 3];
                short depth = (short)Util.GetWord(mFileData, elemOff + 4, 2, false);
                short left = (short)Util.GetWord(mFileData, elemOff + 6, 2, false);
                short top = (short)Util.GetWord(mFileData, elemOff + 8, 2, false);
                short right = (short)Util.GetWord(mFileData, elemOff + 10, 2, false);
                short bottom = (short)Util.GetWord(mFileData, elemOff + 12, 2, false);

                if (depth == 0x0060) {
                    // Offset it to give it a sense of depth.  We really only want to do this
                    // for the 100-point enemy ship.
                    mAppRef.DebugLog("Offsetting for depth");
                    left -= 64;
                    right -= 64;
                    top -= 64;
                    bottom -= 64;
                }

                switch (kind) {
                    case 0x00:      // point
                        // right/bottom not used, ignore them
                        elems.Add(new EpochPoint(mAppRef, colorA, colorB, colorFreq,
                            left, top, left, top));
                        break;
                    case 0x01:      // horizontal line
                        // bottom not used, ignore it
                        elems.Add(new EpochLine(mAppRef, colorA, colorB, colorFreq,
                            left, top, right, top));
                        break;
                    case 0x80:      // rect
                        elems.Add(new EpochRect(mAppRef, colorA, colorB, colorFreq,
                            left, top, right, bottom));
                        break;
                    case 0xff:      // vertical line
                        // right not used, ignore it
                        elems.Add(new EpochLine(mAppRef, colorA, colorB, colorFreq,
                            left, top, left, bottom));
                        break;
                    default:
                        mAppRef.ReportError("Unexpected element type " + kind.ToString("x2"));
                        break;
                }

                elemOff += ELEM_LEN;
            }

            //
            // Compute the bounds and scale factor.
            //
            int minX, minY, maxX, maxY;
            minX = minY = 65536;
            maxX = maxY = -65536;
            foreach (Element elem in elems) {
                elem.UpdateMinMax(ref minX, ref minY, ref maxX, ref maxY);
            }
            int maxDiff = Math.Max(
                Math.Max(Math.Abs(minX), Math.Abs(maxX)),
                Math.Max(Math.Abs(minY), Math.Abs(maxY)));
            double scale;
            if (maxDiff == 0) {
                scale = 1.0;
            } else {
                scale = (double)(BITMAP_DIM - 2) / 2 / (double)maxDiff;
            }
            if (VERBOSE) {
                mAppRef.DebugLog("off=+" + offset.ToString("x6") +
                    " minX=" + minX + " minY=" + minY + " maxX=" + maxX + " maxY=" + maxY +
                    " maxDiff=" + maxDiff + " scale=" + scale);
            }

            VisBitmap8 vb = new VisBitmap8(BITMAP_DIM, BITMAP_DIM);
            SetHiResPalette(vb);

            sMotleyColor = HiResColors.Black0;

            if (outlineMode) {
                foreach (Element elem in elems) {
                    if (elem is EpochRect) {
                        elem.Draw(vb, ColorMode.Dim, scale, false);
                    }
                }
            }

            foreach (Element elem in elems) {
                elem.Draw(vb, scrambleColors ? ColorMode.Motley : ColorMode.Standard, scale,
                    outlineMode);
            }

            return vb;
        }

        /// <summary>
        /// Map hi-res colors to palette entries.
        /// </summary>
        private enum HiResColors : byte {
            Black0 = 1,
            Green = 3,
            Purple = 4,
            White0 = 2,
            Black1 = 1,
            Orange = 5,
            Blue = 6,
            White1 = 2,

            Whoops = Blue + DARK_SHIFT + 1
        }
        const int DARK_SHIFT = 6;

        private static void SetHiResPalette(VisBitmap8 vb) {
            // These don't match directly to hi-res color numbers because we want to
            // avoid adding black/white twice.  The colors correspond to Apple IIgs RGB
            // monitor output.
            vb.AddColor(0, 0, 0, 0);                // 0=transparent

            int black = Util.MakeARGB(0xff, 0x00, 0x00, 0x00);
            int white = Util.MakeARGB(0xff, 0xff, 0xff, 0xff);
            int green = Util.MakeARGB(0xff, 0x11, 0xdd, 0x00);
            int purple = Util.MakeARGB(0xff, 0xdd, 0x22, 0xdd);
            int orange = Util.MakeARGB(0xff, 0xff, 0x66, 0x00);
            int blue = Util.MakeARGB(0xff, 0x22, 0x22, 0xff);

            vb.AddColor(black);             // 1=black0/black1
            vb.AddColor(white);             // 2=white0/white1
            vb.AddColor(green);             // 3=green
            vb.AddColor(purple);            // 4=purple
            vb.AddColor(orange);            // 5=orange
            vb.AddColor(blue);              // 6=blue

            // Repeat colors 1-6, darkened.
            const float darkFactor = 0.6f;
            vb.AddColor(0xff, 0x40, 0x40, 0x40);            // 7=dark grey
            vb.AddColor(MultColor(white, darkFactor));      // 8=light grey
            vb.AddColor(MultColor(green, darkFactor));      // 9=dark green
            vb.AddColor(MultColor(purple, darkFactor));     // 10=dark purple
            vb.AddColor(MultColor(orange, darkFactor));     // 11=dark orange
            vb.AddColor(MultColor(blue, darkFactor));       // 12=dark blue

            vb.AddColor(0xff, 0xff, 0xff, 0x00);            // 13=whoops! (bright yellow)
        }

        /// <summary>
        /// Converts an Epoch color to a hi-res color constant.
        /// </summary>
        private static HiResColors GetHiResColor(byte epochColor, ColorMode cmode) {
            HiResColors color;

            if (cmode == ColorMode.Motley) {
                color = sMotleyColor;
                int nextColor = (int)sMotleyColor + 1;
                if (nextColor > (int)HiResColors.Blue) {
                    nextColor = (int)HiResColors.Black0;
                }
                sMotleyColor = (HiResColors)nextColor;
                return color;
            }

            switch (epochColor) {
                case 0x00:      // black
                    color = HiResColors.Black0;
                    break;
                case 0x10:      // white
                    color = HiResColors.White0;
                    break;
                case 0x20:      // green
                    color = HiResColors.Green;
                    break;
                case 0x30:      // purple
                    color = HiResColors.Purple;
                    break;
                case 0x40:      // orange
                    color = HiResColors.Orange;
                    break;
                case 0x50:      // blue
                    color = HiResColors.Blue;
                    break;
                case 0x60:      // white
                    color = HiResColors.White0;
                    break;
                default:        // whoops
                    color = HiResColors.Whoops;
                    break;
            }

            if (cmode == ColorMode.Dim) {
                color = (HiResColors)((int)color + DARK_SHIFT);
            }
            return color;
        }

        /// <summary>
        /// Multiplies an ARGB value by a constant.
        /// </summary>
        private static int MultColor(int color, float mult) {
            int newA = (color >> 24) & 0xff;
            int newR = (int)(((color >> 16) & 0xff) * mult);
            int newG = (int)(((color >> 8) & 0xff) * mult);
            int newB = (int)((color & 0xff) * mult);
            int res = (newA << 24) | (newR << 16) | (newG << 8) | newB;
            return res;
        }
    }
}
