﻿/*
 * Copyright 2020 faddenSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Security.Policy;

using PluginCommon;

namespace PhantomsFive {
    /// <summary>
    /// Visualizer for Phantoms Five bitmaps and other data.  This is a horribly hacked-up
    /// version of Apple/VisHiRes.cs.
    /// </summary>
    public class VisP5Gfx : MarshalByRefObject, IPlugin, IPlugin_Visualizer {
        // IPlugin
        public string Identifier {
            get { return "Apple II Phantoms Five Graphic Visualizer"; }
        }
        private IApplication mAppRef;
        private byte[] mFileData;
        private AddressTranslate mAddrTrans;

        // Visualization identifiers; DO NOT change or projects that use them will break.
        private const string VIS_P5_BITMAP = "p5-bitmap";
        private const string VIS_P5_PALIGN_BITMAP = "p5-palign-bitmap";
        private const string VIS_P5_TERRAIN = "p5-terrain";
        private const string VIS_P5_SHATTER = "p5-shatter";

        private const string P_OFFSET = "offset";
        private const string P_BYTE_WIDTH = "byteWidth";
        private const string P_HEIGHT = "height";
        private const string P_COL_STRIDE = "colStride";
        private const string P_ROW_STRIDE = "rowStride";
        private const string P_CELL_STRIDE = "cellStride";
        private const string P_IS_COLOR = "isColor";
        private const string P_IS_FIRST_ODD = "isFirstOdd";
        private const string P_IS_HIGH_BIT_FLIPPED = "isHighBitFlipped";
        private const string P_COLOR_CONV_MODE = "colorConvMode";
        private const string P_ITEM_BYTE_WIDTH = "itemByteWidth";
        private const string P_ITEM_HEIGHT = "itemHeight";
        private const string P_COUNT = "count";
        private const string P_IS_DOUBLED = "isDoubled";
        private const string P_LINE_ALIGNMENT = "lineAlignment";

        private const int MAX_DIM = 4096;

        // Visualization descriptors.
        private VisDescr[] mDescriptors = new VisDescr[] {
            new VisDescr(VIS_P5_BITMAP, "Phantoms Five Bitmap", VisDescr.VisType.Bitmap,
                new VisParamDescr[] {
                    new VisParamDescr("File offset (hex)",
                        P_OFFSET, typeof(int), 0, 0x00ffffff, VisParamDescr.SpecialMode.Offset, 0),
                    new VisParamDescr("Width (in bytes)",
                        P_BYTE_WIDTH, typeof(int), 1, 256, 0, 1),
                    new VisParamDescr("Height",
                        P_HEIGHT, typeof(int), 1, 1024, 0, 1),
                    new VisParamDescr("Column stride (bytes)",
                        P_COL_STRIDE, typeof(int), 0, 1024, 0, 0),
                    new VisParamDescr("Row stride (bytes)",
                        P_ROW_STRIDE, typeof(int), 0, 1024, 0, 0),
                    new VisParamDescr("Line alignment",
                        P_LINE_ALIGNMENT, typeof(int), 0, 256, 0, 0),
                    new VisParamDescr("Color",
                        P_IS_COLOR, typeof(bool), 0, 0, 0, true),
                    new VisParamDescr("First col odd",
                        P_IS_FIRST_ODD, typeof(bool), 0, 0, 0, false),
                    new VisParamDescr("High bit flipped",
                        P_IS_HIGH_BIT_FLIPPED, typeof(bool), 0, 0, 0, false),
                    new VisParamDescr("Doubled lines",
                        P_IS_DOUBLED, typeof(bool), 0, 0, 0, false),
                }),
            new VisDescr(VIS_P5_TERRAIN, "Phantoms Five Annotated Terrain", VisDescr.VisType.Bitmap,
                new VisParamDescr[] {
                    new VisParamDescr("File offset (hex)",
                        P_OFFSET, typeof(int), 0, 0x00ffffff, VisParamDescr.SpecialMode.Offset, 0),
                }),
            new VisDescr(VIS_P5_SHATTER, "Phantoms Five Windshield Shatter", VisDescr.VisType.Bitmap,
                new VisParamDescr[] {
                    new VisParamDescr("File offset (hex)",
                        P_OFFSET, typeof(int), 0, 0x00ffffff, VisParamDescr.SpecialMode.Offset, 0),
                }),
        };


        // IPlugin
        public void Prepare(IApplication appRef, byte[] fileData, AddressTranslate addrTrans) {
            mAppRef = appRef;
            mFileData = fileData;
            mAddrTrans = addrTrans;
        }

        // IPlugin
        public void Unprepare() {
            mAppRef = null;
            mFileData = null;
            mAddrTrans = null;
        }

        // IPlugin_Visualizer
        public VisDescr[] GetVisGenDescrs() {
            // We're using a static set, but it could be generated based on file contents.
            // Confirm that we're prepared.
            if (mFileData == null) {
                return null;
            }
            return mDescriptors;
        }

        // IPlugin_Visualizer
        public IVisualization2d Generate2d(VisDescr descr,
                ReadOnlyDictionary<string, object> parms) {
            switch (descr.Ident) {
                case VIS_P5_BITMAP:
                    return GenerateBitmap(parms);
                case VIS_P5_TERRAIN:
                    return GenerateTerrain(parms);
                case VIS_P5_SHATTER:
                    return GenerateShatter(parms);
                default:
                    mAppRef.ReportError("Unknown ident " + descr.Ident);
                    return null;
            }
        }

        private IVisualization2d GenerateBitmap(ReadOnlyDictionary<string, object> parms) {
            int offset = Util.GetFromObjDict(parms, P_OFFSET, 0);
            int byteWidth = Util.GetFromObjDict(parms, P_BYTE_WIDTH, 1); // width ignoring colStride
            int height = Util.GetFromObjDict(parms, P_HEIGHT, 1);
            int colStride = Util.GetFromObjDict(parms, P_COL_STRIDE, 0);
            int rowStride = Util.GetFromObjDict(parms, P_ROW_STRIDE, 0);
            bool isColor = Util.GetFromObjDict(parms, P_IS_COLOR, true);
            bool isFirstOdd = Util.GetFromObjDict(parms, P_IS_FIRST_ODD, false);
            bool isHighBitFlipped = Util.GetFromObjDict(parms, P_IS_HIGH_BIT_FLIPPED, false);
            int colorConvMode = !isColor ? (int)ColorMode.Mono :
                Util.GetFromObjDict(parms, P_COLOR_CONV_MODE, (int)ColorMode.SimpleColor);
            bool isDoubled = Util.GetFromObjDict(parms, P_IS_DOUBLED, false);
            int lineAlignment = Util.GetFromObjDict(parms, P_LINE_ALIGNMENT, 0);

            // We allow the stride entries to be zero to indicate a "dense" bitmap.
            if (colStride == 0) {
                colStride = 1;
            }
            if (rowStride == 0) {
                rowStride = byteWidth * colStride;
            }

            if (offset < 0 || offset >= mFileData.Length ||
                    byteWidth <= 0 || byteWidth > MAX_DIM ||
                    height <= 0 || height > MAX_DIM) {
                // the UI should flag these based on range (and ideally wouldn't have called us)
                mAppRef.ReportError("Invalid parameter");
                return null;
            }
            if (colStride <= 0 || colStride > MAX_DIM) {
                mAppRef.ReportError("Invalid column stride");
                return null;
            }
            if (rowStride < 1 || rowStride > MAX_DIM) {
                mAppRef.ReportError("Invalid row stride");
                return null;
            }

            //int lastOffset = offset + rowStride * height - (colStride - 1) - 1;
            //if (lastOffset >= mFileData.Length) {
            //    mAppRef.ReportError("Bitmap runs off end of file (last offset +" +
            //        lastOffset.ToString("x6") + ")");
            //    return null;
            //}

            // Double-height bitmap because every other line is black.
            VisBitmap8 vb = new VisBitmap8(byteWidth * 7, height + (isDoubled ? height : 0));
            SetHiResPalette(vb);

            try {
                RenderBitmap(mFileData, offset, byteWidth, height, colStride, rowStride,
                    (ColorMode)colorConvMode, isFirstOdd, isHighBitFlipped, vb, 0, 0, isDoubled,
                    lineAlignment);
            } catch (AddressTranslateException ex) {
                mAppRef.ReportError(ex.Message);
                return null;
            }
            return vb;
        }

        private IVisualization2d GenerateTerrain(ReadOnlyDictionary<string, object> parms) {
            int offset = Util.GetFromObjDict(parms, P_OFFSET, 0);
            if (offset < 0 || offset >= mFileData.Length) {
                mAppRef.ReportError("Invalid parameter");
                return null;
            }

            // Hard-code everything else.
            int byteWidth = 40;
            int height = 345;
            int colStride = 1;
            int rowStride = byteWidth * colStride;
            ColorMode colorConvMode = ColorMode.SimpleColor;
            bool isFirstOdd = false;
            bool isHighBitFlipped = false;

            VisBitmap8 vb = new VisBitmap8(byteWidth * 7, height * 2);
            SetHiResPalette(vb);

            try {
                // Render terrain on odd rows.
                RenderBitmap(mFileData, offset, byteWidth, height, colStride, rowStride,
                    colorConvMode, isFirstOdd, isHighBitFlipped, vb, 0, 1, true, 0);
            } catch (AddressTranslateException ex) {
                mAppRef.ReportError(ex.Message);
                return null;
            }

            // Now render the scoring annotation on the even rows, using the bitmap data from
            // the odd rows and a few special entries.
            int addr = mAddrTrans.OffsetToAddress(offset);
            for (int row = 0; row < height; row++) {
                int by = row * 2;       // draw on even row, starting from zero
                byte firstVal = mAddrTrans.GetDataAtAddress(mFileData, offset,
                        addr + row * rowStride);
                byte lastVal = mAddrTrans.GetDataAtAddress(mFileData, offset,
                        addr + row * rowStride + byteWidth - 1);
                for (int col = 0; col < byteWidth; col++) {
                    HiResColors annoColor;

                    byte terr = mAddrTrans.GetDataAtAddress(mFileData, offset,
                        addr + row * rowStride + col);

                    // Basic score comes from terrain color.
                    if (terr == 0x00) {
                        annoColor = HiResColors.LowValue;
                    } else if (terr < 0x7f) {
                        annoColor = HiResColors.MedValue;
                    } else {
                        annoColor = HiResColors.HighValue;
                    }

                    // Now check special targets.  Technically these are *in addition* to the
                    // above, but that's awkward to display color-wise.  Special areas are
                    // identified vertically by a stripe on the left or right edge of the screen,
                    // then horizontally with hard-coded constants.
                    if (lastVal == 0x01) {
                        // Check POW camp.
                        if (col >= 0x0a && col < 0x16) {
                            annoColor = HiResColors.BadSpecial;
                        }
                    } else if (firstVal == 0x01) {
                        // Check fuel dump.
                        if (col >= 0x0c && col <= 0x0d) {
                            annoColor = HiResColors.GoodSpecial;
                        }
                    } else if (firstVal == 0x81) {
                        // Check hospital.
                        if (col >= 0x0e && col < 0x14) {
                            annoColor = HiResColors.BadSpecial;
                        }
                    } else if (firstVal == 0x82) {
                        // Check HQ.
                        if (col >= 0x0e && col < 0x14) {
                            annoColor = HiResColors.GoodSpecial;
                        }
                    }

                    int bx = col * 7;
                    for (int i = 0; i < 7; i++) {
                        vb.SetPixelIndex(bx + i, by, (byte)annoColor);
                    }
                }
            }

            return vb;
        }

        private IVisualization2d GenerateShatter(ReadOnlyDictionary<string, object> parms) {
            int offset = Util.GetFromObjDict(parms, P_OFFSET, 0);
            if (offset < 0 || offset >= mFileData.Length) {
                mAppRef.ReportError("Invalid parameter");
                return null;
            }

            // Hard-code everything else.
            int byteWidth = 40;
            int height = 192;
            int colStride = 1;
            int rowStride = byteWidth * colStride;
            ColorMode colorConvMode = ColorMode.SimpleColor;
            bool isFirstOdd = false;
            bool isHighBitFlipped = false;

            // Allocate a buffer for hi-res data, and color it blue.
            byte[] data = new byte[8192];
            for (int i = 0; i < data.Length; i += 2) {
                data[i] = 0xd5;
                data[i + 1] = 0xaa;
            }

            // Apply shatter data.
            int srcOff = offset;
            int dstOff = 0;
            while (srcOff < mFileData.Length && dstOff < 8192) {
                byte val = mFileData[srcOff++];
                if (val <= 0x7f) {
                    data[dstOff++] ^= val;
                } else if (val < 0xff) {
                    dstOff += (val & 0x7f);
                } else /*0xff*/ {
                    break;
                }
            }

            // Linearize the data.
            byte[] buf = new byte[byteWidth * height];
            int outIdx = 0;
            for (int row = 0; row < height; row++) {
                // If row is ABCDEFGH, we want pppFGHCD EABAB000 (where p would be $20/$40).
                int low = ((row & 0xc0) >> 1) | ((row & 0xc0) >> 3) | ((row & 0x08) << 4);
                int high = ((row & 0x07) << 2) | ((row & 0x30) >> 4);
                int rowAddr = ((high << 8) | low);

                for (int col = 0; col < byteWidth; col++) {
                    buf[outIdx++] = data[rowAddr + col];
                }
            }

            // Black out the top 8 lines, where the status bar would be.
            for (int i = 0; i < 8 * byteWidth; i++) {
                buf[i] = 0;
            }

            // Convert to bitmap.
            VisBitmap8 vb = new VisBitmap8(byteWidth * 7, height);
            SetHiResPalette(vb);

            RenderBitmap(buf, -1, byteWidth, height, colStride, rowStride,
                colorConvMode, isFirstOdd, isHighBitFlipped, vb, 0, 0, false, 0);
            return vb;
        }


        private enum ColorMode { Mono, SimpleColor };

        /// <summary>
        /// Renders bitmap data.
        /// </summary>
        /// <param name="data">Data source, typically the file data.</param>
        /// <param name="srcOffset">Offset into data of the first byte.</param>
        /// <param name="byteWidth">Width, in bytes, of the data to render.  Each byte
        ///   represents 7 pixels in the output (more or less).</param>
        /// <param name="height">Height, in lines, of the data to render.</param>
        /// <param name="colStride">Column stride.  The number of bytes used to hold each
        ///   byte of data.  Must be >= 1.</param>
        /// <param name="rowStride">Row stride.  The number of bytes used to hold each row
        ///   of data.  Must be >= (colStride * byteWidth - (colStride - 1)).</param>
        /// <param name="colorMode">Color conversion mode.</param>
        /// <param name="isFirstOdd">If true, render as if we're starting on an odd column.
        ///   This affects the colors.</param>
        /// <param name="isHighBitFlipped">If true, render as if the high bit has the
        ///   opposite value.  This affects the colors.</param>
        /// <param name="vb">Output bitmap object.</param>
        /// <param name="xstart">Initial X position in the output.</param>
        /// <param name="ystart">Initial Y position in the output.</param>
        /// <param name="doubleSpace">If true, add extra line.</param>
        /// <param name="lineAlignment">Handle weird mid-bitmap alignment thing.</param>
        private void RenderBitmap(byte[] data, int srcOffset, int byteWidth, int height,
                int colStride, int rowStride, ColorMode colorMode, bool isFirstOdd,
                bool isHighBitFlipped, VisBitmap8 vb, int xstart, int ystart,
                bool doubleSpace, int lineAlignment) {
            int addr = srcOffset < 0 ? -1 : mAddrTrans.OffsetToAddress(srcOffset);
            int bx = xstart;
            int by = ystart;
            switch (colorMode) {
                case ColorMode.Mono: {
                        // Since we're not displaying this we don't need to worry about
                        // half-pixel shifts, and can just convert 7 bits to pixels.
                        for (int row = 0; row < height; row++) {
                            int colIdx = 0;
                            for (int col = 0; col < byteWidth; col++) {
                                byte val = mAddrTrans.GetDataAtAddress(data, srcOffset,
                                    addr + colIdx);
                                for (int bit = 0; bit < 7; bit++) {
                                    if ((val & 0x01) == 0) {
                                        vb.SetPixelIndex(bx, by, (int)HiResColors.Black0);
                                    } else {
                                        vb.SetPixelIndex(bx, by, (int)HiResColors.White0);
                                    }
                                    val >>= 1;
                                    bx++;
                                }
                                colIdx += colStride;
                            }
                            bx = xstart;
                            by++;
                            addr += rowStride;

                            if (lineAlignment != 0) {
                                int mask = lineAlignment - 1;

                                // See if next line fits within alignment restrictions.
                                if ((addr & ~mask) != ((addr + rowStride - 1) & ~mask)) {
                                    mAppRef.DebugLog("Adjusting addr " + addr.ToString("x4"));
                                    addr = (addr + lineAlignment - 1) & ~mask;
                                    mAppRef.DebugLog(" New addr " + addr.ToString("x4"));
                                }
                            }

                            if (doubleSpace && by < vb.Height) {
                                // draw an empty row
                                for (int col = 0; col < byteWidth * 7; col++) {
                                    vb.SetPixelIndex(bx + col, by, (int)HiResColors.Black0);
                                }
                                by++;
                            }
                        }
                    }
                    break;
                case ColorMode.SimpleColor: {
                        // Straightforward conversion, with no funky border effects.  This
                        // represents an idealized version of the hardware.

                        // Bits for every byte, plus a couple of "fake" bits on the ends so
                        // we don't have to throw range-checks everywhere.
                        const int OVER = 2;
                        bool[] lineBits = new bool[OVER + byteWidth * 7 + OVER];
                        bool[] hiFlags = new bool[OVER + byteWidth * 7 + OVER];
                        for (int row = 0; row < height; row++) {
                            // Unravel the bits.  Note we do each byte "backwards", i.e. the
                            // low bit (which is generally considered to be on the right) is
                            // the leftmost pixel.
                            int idx = OVER;     // start past "fake" bits
                            int colIdx = 0;
                            for (int col = 0; col < byteWidth; col++) {
                                byte val;
                                if (addr < 0) {
                                    val = data[row * byteWidth + colIdx];
                                } else {
                                    val = mAddrTrans.GetDataAtAddress(data, srcOffset,
                                        addr + colIdx);
                                }
                                bool hiBitSet = (val & 0x80) != 0;

                                for (int bit = 0; bit < 7; bit++) {
                                    hiFlags[idx] = hiBitSet ^ isHighBitFlipped;
                                    lineBits[idx] = (val & 0x01) != 0;
                                    idx++;
                                    val >>= 1;
                                }
                                colIdx += colStride;
                            }

                            // Convert to color.
                            int lastBit = byteWidth * 7;
                            for (idx = OVER; idx < lastBit + OVER; idx++) {
                                int colorShift = hiFlags[idx] ? 2 : 0;
                                if (lineBits[idx] && (lineBits[idx - 1] || lineBits[idx + 1])) {
                                    // [X]11 or [1]1X; two 1s in a row is always white
                                    vb.SetPixelIndex(bx++, by, (byte)HiResColors.White0);
                                } else if (lineBits[idx]) {
                                    // [0]10, color pixel
                                    bool isOdd = ((idx & 0x01) != 0) ^ isFirstOdd;
                                    if (isOdd) {
                                        vb.SetPixelIndex(bx++, by,
                                                (byte)((int)HiResColors.Green + colorShift));
                                    } else {
                                        vb.SetPixelIndex(bx++, by,
                                                (byte)((int)HiResColors.Purple + colorShift));
                                    }
                                } else if (lineBits[idx - 1] && lineBits[idx + 1]) {
                                    // [1]01, keep color going
                                    bool isOdd = ((idx & 0x01) != 0) ^ isFirstOdd;
                                    if (isOdd) {
                                        vb.SetPixelIndex(bx++, by,
                                                (byte)((int)HiResColors.Purple + colorShift));
                                    } else {
                                        vb.SetPixelIndex(bx++, by,
                                                (byte)((int)HiResColors.Green + colorShift));
                                    }
                                } else {
                                    // [0]0X or [X]01
                                    vb.SetPixelIndex(bx++, by, (byte)HiResColors.Black0);
                                }
                            }

                            // move to next row
                            bx = xstart;
                            by++;
                            if (addr >= 0) {
                                addr += rowStride;

                                if (lineAlignment != 0) {
                                    int mask = lineAlignment - 1;

                                    // See if next line fits within alignment restrictions.
                                    if ((addr & ~mask) != ((addr + rowStride - 1) & ~mask)) {
                                        mAppRef.DebugLog("Adjusting addr " + addr.ToString("x4"));
                                        addr = (addr + lineAlignment - 1) & ~mask;
                                        mAppRef.DebugLog(" New addr " + addr.ToString("x4"));
                                    }
                                }
                            }

                            // draw an empty row
                            if (doubleSpace && by < vb.Height) {
                                for (int col = 0; col < byteWidth * 7; col++) {
                                    vb.SetPixelIndex(bx + col, by, (int)HiResColors.Black0);
                                }
                                by++;
                            }
                        }
                    }
                    break;
                default:
                    // just leave the bitmap empty
                    mAppRef.ReportError("Unknown ColorMode " + colorMode);
                    break;
            }
        }

        /// <summary>
        /// Map hi-res colors to palette entries.
        /// </summary>
        private enum HiResColors : byte {
            Black0      = 1,
            Green       = 3,
            Purple      = 4,
            White0      = 2,
            Black1      = 1,
            Orange      = 5,
            Blue        = 6,
            White1      = 2,

            // a few additional entries for scoring annotations
            GoodSpecial = 7,
            BadSpecial  = 8,
            LowValue    = 1,    // re-use black
            MedValue    = 9,
            HighValue   = 10,
        }

        private void SetHiResPalette(VisBitmap8 vb) {
            // These don't match directly to hi-res color numbers because we want to
            // avoid adding black/white twice.  The colors correspond to Apple IIgs RGB
            // monitor output.
            vb.AddColor(0, 0, 0, 0);                // 0=transparent
            vb.AddColor(0xff, 0x00, 0x00, 0x00);    // 1=black0/black1
            vb.AddColor(0xff, 0xff, 0xff, 0xff);    // 2=white0/white1
            vb.AddColor(0xff, 0x11, 0xdd, 0x00);    // 3=green
            vb.AddColor(0xff, 0xdd, 0x22, 0xdd);    // 4=purple
            vb.AddColor(0xff, 0xff, 0x66, 0x00);    // 5=orange
            vb.AddColor(0xff, 0x22, 0x22, 0xff);    // 6=blue

            vb.AddColor(0xff, 0x00, 0xff, 0x00);    // 7=good special (green)
            vb.AddColor(0xff, 0xff, 0x00, 0x00);    // 8=bad special (red)
            vb.AddColor(0xff, 0x00, 0x80, 0x80);    // 10=medium value (Teal)
            vb.AddColor(0xff, 0xff, 0xd7, 0x00);    // 11=high value (Gold)
        }
    }
}
