﻿/*
 * Copyright 2020 faddenSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
using System;
using System.Collections.ObjectModel;

using PluginCommon;

namespace Stellar7 {
    /// <summary>
    /// Visualizer for Stellar 7 meshes.
    /// </summary>
    public class Stellar7Mesh : MarshalByRefObject, IPlugin, IPlugin_Visualizer_v2 {
        // IPlugin
        public string Identifier {
            get { return "Stellar 7 Mesh Visualizer"; }
        }
        private IApplication mAppRef;
        private byte[] mFileData;

        // Visualization identifiers; DO NOT change or projects that use them will break.
        private const string VIS_GEN_STELLAR7_MESH = "stellar7-mesh";

        private const string P_OFFSET = "offset";

        // Visualization descriptors.
        private VisDescr[] mDescriptors = new VisDescr[] {
            new VisDescr(VIS_GEN_STELLAR7_MESH, "Stellar 7 Mesh", VisDescr.VisType.Wireframe,
                new VisParamDescr[] {
                    new VisParamDescr("File offset (hex)",
                        P_OFFSET, typeof(int), 0, 0x00ffffff, VisParamDescr.SpecialMode.Offset, 0),
                }),
        };


        // IPlugin
        public void Prepare(IApplication appRef, byte[] fileData, AddressTranslate addrTrans) {
            mAppRef = appRef;
            mFileData = fileData;
        }

        // IPlugin
        public void Unprepare() {
            mAppRef = null;
            mFileData = null;
        }

        // IPlugin_Visualizer
        public VisDescr[] GetVisGenDescrs() {
            if (mFileData == null) {
                return null;
            }
            return mDescriptors;
        }

        // IPlugin_Visualizer
        public IVisualization2d Generate2d(VisDescr descr,
                ReadOnlyDictionary<string, object> parms) {
            mAppRef.ReportError("2d not supported");
            return null;
        }

        // IPlugin_Visualizer_v2
        public IVisualizationWireframe GenerateWireframe(VisDescr descr,
                ReadOnlyDictionary<string, object> parms) {
            switch (descr.Ident) {
                case VIS_GEN_STELLAR7_MESH:
                    return GenerateWireframe(parms);
                default:
                    mAppRef.ReportError("Unknown ident " + descr.Ident);
                    return null;
            }
        }

        private IVisualizationWireframe GenerateWireframe(ReadOnlyDictionary<string, object> parms) {
            int offset = Util.GetFromObjDict(parms, P_OFFSET, 0);

            const int END_MARKER = 0x80;

            if (offset < 0 || offset >= mFileData.Length) {
                // should be caught by editor
                mAppRef.ReportError("Invalid parameter");
                return null;
            }

            VisWireframe vw = new VisWireframe();
            try {
                // Gather vertices.  The Y coordinate seems to be positioned with the
                // player view height already taken into account, so we leave it alone.
                int vertexCount = 0;
                while (mFileData[offset] != END_MARKER) {
                    byte dist = mFileData[offset++];
                    byte angle = mFileData[offset++];
                    int yc = (sbyte)mFileData[offset++];

                    double rads = (angle / 128.0) * Math.PI;
                    double xc = Math.Cos(rads) * dist;
                    double zc = Math.Sin(rads) * dist;

                    vw.AddVertex((float)xc, (float)yc, (float)zc);
                    vertexCount++;
                }
                offset++;

                // Gather edges.
                int edgeCount = 0;
                while (mFileData[offset] != END_MARKER) {
                    byte v0 = mFileData[offset++];
                    byte v1 = mFileData[offset++];

                    vw.AddEdge(v0, v1);
                    edgeCount++;
                }

                //mAppRef.DebugLog("S7: verts=" + vertexCount + " edges=" + edgeCount);
            } catch (IndexOutOfRangeException) {
                // assume it was our file data access that caused the failure
                mAppRef.ReportError("Ran off end of file");
                return null;
            }

            string msg;
            if (!vw.Validate(out msg)) {
                mAppRef.ReportError("Data error: " + msg);
                return null;
            }

            return vw;
        }
    }
}
